/*
 * Function library for ILI9341 display and XPT2046 touch
 * Designed for Micromite
 */


#include <xc.h>
#include "backpack.h"
#define constrain(amt,low,high) ((amt)<(low)?(low):((amt)>(high)?(high):(amt)))

void delay(uint32_t t){
  uint32_t n;
  t=t*2400;                 //~1ms, may change depending on optimisation level, try 2400 for L0, 6k for L1
  for (n=0; n<t;++n){Nop();}
}

void backpackinit(){
    //LCD backpack is CS on 6 (RB2), DC on 2 (RA0), RST on 23 (RB12)    
    //TOUCHCS on 7 (RB3)
    //no analog
    ANSELBbits.ANSB2=0;
    ANSELAbits.ANSA0=0;
    ANSELBbits.ANSB12=0;    
    ANSELBbits.ANSB3=0;
    ANSELBbits.ANSB15=0;//backlight    
    //outputs
    TRISBbits.TRISB2=0;
    TRISAbits.TRISA0=0;
    TRISBbits.TRISB12=0;
    TRISBbits.TRISB3=0;
    TRISBbits.TRISB15=0;//backlight
    //set DC low, CSs high

    //set up SD SPI so it's ready too
    LATBbits.LATB0=1;
    ANSELBbits.ANSB0=0;
    TRISBbits.TRISB0=0;
    
    backpackSPIinit();

    BACKPACK_CS=1;
    BACKPACK_DC=0;
    BACKPACK_TOUCHCS=1;
    BACKPACK_RST=0;
    BACKPACK_BACKLIGHT=1;
    delay(100);
    BACKPACK_RST=1;
    delay(40);
    BACKPACK_CS=0;
    backpackcmd(0xE0); // Positive Gamma Control
    backpackdata(0x00);
    backpackdata(0x03);
    backpackdata(0x09);
    backpackdata(0x08);
    backpackdata(0x16);
    backpackdata(0x0A);
    backpackdata(0x3F);
    backpackdata(0x78);
    backpackdata(0x4C);
    backpackdata(0x09);
    backpackdata(0x0A);
    backpackdata(0x08);
    backpackdata(0x16);
    backpackdata(0x1A);
    backpackdata(0x0F);

    backpackcmd(0XE1); // Negative Gamma Control
    backpackdata(0x00);
    backpackdata(0x16);
    backpackdata(0x19);
    backpackdata(0x03);
    backpackdata(0x0F);
    backpackdata(0x05);
    backpackdata(0x32);
    backpackdata(0x45);
    backpackdata(0x46);
    backpackdata(0x04);
    backpackdata(0x0E);
    backpackdata(0x0D);
    backpackdata(0x35);
    backpackdata(0x37);
    backpackdata(0x0F);

    backpackcmd(0XC0); // Power Control 1
    backpackdata(0x17);
    backpackdata(0x15);

    backpackcmd(0xC1); // Power Control 2
    backpackdata(0x41);

    backpackcmd(0xC5); // VCOM Control
    backpackdata(0x00);
    backpackdata(0x12);
    backpackdata(0x80);

    backpackcmd(0x36); // Memory Access Control
    backpackdata(0x48); // MX, BGR

    backpackcmd(0x3A); // Pixel Interface Format
    backpackdata(0x66); // 18 bit colour for SPI

    backpackcmd(0xB0); // Interface Mode Control
    backpackdata(0x00);

    backpackcmd(0xB1); // Frame Rate Control
    backpackdata(0xA0);

    backpackcmd(0xB4); // Display Inversion Control
    backpackdata(0x02);

    backpackcmd(0xB6); // Display Function Control
    backpackdata(0x02);
    backpackdata(0x02);
    backpackdata(0x3B);

    backpackcmd(0xB7); // Entry Mode Set
    backpackdata(0xC6);

    backpackcmd(0xF7); // Adjust Control 3
    backpackdata(0xA9);
    backpackdata(0x51);
    backpackdata(0x2C);
    backpackdata(0x82);

    backpackcmd(0x11); //Exit Sleep
    delay(120);
    backpackcmd(0x29); //Display on
    delay(25);
    BACKPACK_CS=1;
    backpackrotate(1);
    backpackareaset(0,0,backpackwidth-1,backpackheight-1);
}


void backpackrotate(char a){
    BACKPACK_CS=0;
    switch(a){
        case 1:
            backpackcmd(0x36);           //Memory Access Control 
            backpackdata(0x48);              //1=top is 12 o'clock
            backpackwidth=320;
            backpackheight=480;
            backpackrotation=1;
            break;
        case 2:
            backpackcmd(0x36);           //Memory Access Control 
            backpackdata(0x28);              //2=top is 3 o'clock
            backpackwidth=480;
            backpackheight=320;
            backpackrotation=2;
            break;
        case 3:
            backpackcmd(0x36);           //Memory Access Control 
            backpackdata(0x88);              //3=top is 6 o'clock
            backpackwidth=320;
            backpackheight=480;
            backpackrotation=3;
            break;
        case 4:
            backpackcmd(0x36);           //Memory Access Control 
            backpackdata(0xE8);              //4=top is 9 o'clock
            backpackwidth=480;
            backpackheight=320;
            backpackrotation=4;
            break;
    }
    BACKPACK_CS=1;
}

void backpackSPIinit(){
    //unlock sequence for PPS
    //LCD backpack SPI is MOSI:3(RA1), MISO:14(RB5), SCK: 25(RB14) MOSI/MISO are remappable
    SYSKEY = 0;                 // force lock
    SYSKEY = 0xAA996655;        // unlock sequence
    SYSKEY = 0x556699AA;        //  
    SDI1R=1;//pin 14 SDI1=>RB5/RPB5
    RPA1R=3;//pin 3 RA1/RPA1=>SDO1
    SYSKEY = 0;                 // force lock
    //SPI setup
    SPI1CON=0;                  //reset SPI, sets 8bit data, frame sync off, use PBCLK, SDI in use
    SPI1BUF;                    //read rx buffer
    SPI1CONbits.MSTEN=1;        //Master
    SPI1CONbits.CKE=1;          //edge
    SPI1CONbits.CKP=0;          //polarity
    SPI1CONbits.MSSEN=0;        //manually control SS    
    SPI1CON2=0;                 //clear CON2, disable audio, disable interrupt flags
    SPI1CON2bits.IGNROV=1;      //ignore RX OF
    SPI1CON2bits.IGNTUR=1;      //ignore TX UF
    SPI1BRG=1;                  //10MHz     
    //SPI1BRG=39;                 //19=> 1MHz (default to full speed, slow down for touch)
    SPI1CONbits.ON=1;           //turn on SPI1    
}

void backpackdata(unsigned char d){
    //while(!SPI1STATbits.SPITBE){}       //wait for buffer empty
    while(SPI1STATbits.SPIBUSY){}       //ensure data is sent
    SPI1BUF=d;                          //write
    while(SPI1STATbits.SPIBUSY){}       //avoid early CS rise
    
}


void backpackcmd(unsigned char d){
    //while(!SPI1STATbits.SPITBE){}       //wait for buffer empty
    while(SPI1STATbits.SPIBUSY){}       //ensure data is sent before changing DC
    BACKPACK_DC=0;
    SPI1BUF=d;                          //write
    while(SPI1STATbits.SPIBUSY){}       //ensure data is sent before changing DC
    BACKPACK_DC=1;    
}

unsigned int backpackSPI16(unsigned int n){
    unsigned int r;
    while(SPI1STATbits.SPIBUSY){}       //ensure data is sent
    SPI1BUF=(n)&0xFF;
    while(SPI1STATbits.SPIBUSY){}       //ensure data is sent
    r=(SPI1BUF&0xFF);
    SPI1BUF=(n>>8)&0xFF;
    while(SPI1STATbits.SPIBUSY){}       //ensure data is sent
    r=r|((SPI1BUF&0xFF)<<8);
    return r;    
}

void backpackareaset(int x1,int y1,int x2,int y2){
    if(x2<x1){int i=x1;x1=x2;x2=i;}   //sort x
    if(y2<y1){int i=y1;y1=y2;y2=i;}   //sort y
    BACKPACK_CS=0;
    backpackcmd(42);               //set x bounds  
    backpackdata(x1>>8);
    backpackdata(x1);
    backpackdata(x2>>8);
    backpackdata(x2);
    backpackcmd(43);               //set y bounds
    backpackdata(y1>>8);
    backpackdata(y1);
    backpackdata(y2>>8);
    backpackdata(y2);
    backpackcmd(44);               //drawing data to follow
    BACKPACK_CS=1;
}


void backpackbox(int x1,int y1,int x2,int y2,unsigned int c){
    int x,y;
    if(x2<x1){int i=x1;x1=x2;x2=i;}
    if(y2<y1){int i=y1;y1=y2;y2=i;}
    backpackareaset(x1,y1,x2,y2);
    x2++;
    y2++;
    BACKPACK_CS=0;
    for(x=x1;x<x2;x++){
      for(y=y1;y<y2;y++){
        backpackdata(c>>16);
        backpackdata(c>>8);
        backpackdata(c);  
      }
    }
    BACKPACK_CS=1;
}


void backpackclear(unsigned int c){
  backpackbox(0,0,backpackwidth-1,backpackheight-1,c);
}

void backpackpoint(int x,int y, unsigned int c){
  backpackareaset(x,y,x,y);
  BACKPACK_CS=0;
  backpackdata(c>>16);
  backpackdata(c>>8);
  backpackdata(c);  
  while(SPI1STATbits.SPIBUSY){}       //ensure data is sent (only seems to be an issue with point?)
  BACKPACK_CS=1;
}

void backpackline(int x1,int y1,int x2,int y2, unsigned int c){
  int steps,stepsx,stepsy,xinc,yinc,x,y,d,i;
  stepsx=abs(x1-x2);
  stepsy=abs(y1-y2);
  steps=max(stepsx,stepsy)+1;   //if start and end are the same, there's still 1 point
  xinc=constrain(x2-x1,-1,1);
  yinc=constrain(y2-y1,-1,1);
  x=x1;
  y=y1;  
  if(stepsx>stepsy){
    d=stepsx/2;
    for(i=0;i<steps;i++){
      backpackpoint(x,y,c);
      x=x+xinc;
      d=d+stepsy;
      if(d>stepsx){d=d-stepsx;y=y+yinc;}
    }
  }else{
    d=stepsy/2;
    for(i=0;i<steps;i++){
      backpackpoint(x,y,c);
      y=y+yinc;
      d=d+stepsx;
      if(d>stepsy){d=d-stepsy;x=x+xinc;}
    } 
  }  
}


void backpackhline(int x1,int y1,int x2,unsigned int c){
  int x;
  if(x2<x1){int i=x1;x1=x2;x2=i;}
  backpackareaset(x1,y1,x2,y1);
  x2++;
  BACKPACK_CS=0;
  for(x=x1;x<x2;x++){
    backpackdata(c>>16);
    backpackdata(c>>8);
    backpackdata(c);  
  }
  BACKPACK_CS=1;

}


void backpackvline(int x1,int y1,int y2,unsigned int c){
  int y;
  if(y2<y1){int i=y1;y1=y2;y2=i;}
  backpackareaset(x1,y1,x1,y2);
  y2++;
  BACKPACK_CS=0;
  for(y=y1;y<y2;y++){
    backpackdata(c>>16);
    backpackdata(c>>8);
    backpackdata(c);  
  }
  BACKPACK_CS=1;
}


int backpacktouchxraw(){
  long n=0;
  int k;
  int oldSPI1BRG;
  oldSPI1BRG=SPI1BRG;
  SPI1BRG=39;
  BACKPACK_TOUCHCS=0;
  backpackdata(0x91);
  n=backpackSPI16(0x91);   //ignore first
  for(k=0;k<TOUCH_OVERSAMPLE-1;k++){
    n=n+backpackSPI16(0x91);
  }
  n=n+backpackSPI16(0x90);   //and one more
  BACKPACK_TOUCHCS=1;
  SPI1BRG=oldSPI1BRG;
  return (n/TOUCH_OVERSAMPLE)>>4;      //only 12 bits resolution
}


int backpacktouchyraw(){
  long n=0;
  int k;
  int oldSPI1BRG;
  oldSPI1BRG=SPI1BRG;
  SPI1BRG=39;
  BACKPACK_TOUCHCS=0;
  backpackdata(0xD1);
  n=backpackSPI16(0xD1);   //ignore first
  for(k=0;k<TOUCH_OVERSAMPLE-1;k++){
    n=n+backpackSPI16(0xD1);
  }
  n=n+backpackSPI16(0xD0);   //and one more
  BACKPACK_TOUCHCS=1;
  SPI1BRG=oldSPI1BRG;
  return (n/TOUCH_OVERSAMPLE)>>4;      //only 12 bits resolution
}


int backpacktouchzraw(){
  int n, oldSPI1BRG;
  oldSPI1BRG=SPI1BRG;
  SPI1BRG=39;
  BACKPACK_TOUCHCS=0;
  backpackdata(0xB1);
  n=backpackSPI16(0xB1);   //ignore first
  n=backpackSPI16(0xB1);
  n=backpackSPI16(0xB0);
  BACKPACK_TOUCHCS=1;
  SPI1BRG=oldSPI1BRG;
  return n>>4;      //only 12 bits resolution
}

int backpackscaletouchx(){      //correct for rotate=2
   long u,v;
   if(backpacktouchzraw()<Z_TOUCH_THRESHOLD){return -1;} //no touch    
   u=backpacktouchxraw();
   v=backpacktouchyraw();
   if(backpacktouchzraw()<Z_TOUCH_THRESHOLD){return -1;} //touch lifted
   return (TOUCH_A*u+TOUCH_B*v+TOUCH_C)>>16;
}

int backpackscaletouchy(){      //correct for rotate=2
   long u,v;
   if(backpacktouchzraw()<Z_TOUCH_THRESHOLD){return -1;} //no touch    
   u=backpacktouchxraw();
   v=backpacktouchyraw();
   if(backpacktouchzraw()<Z_TOUCH_THRESHOLD){return -1;} //touch lifted
   return (TOUCH_D*u+TOUCH_E*v+TOUCH_F)>>16;    
}

int backpacktouchx(){
  int n;
  switch(backpackrotation){
    case 1:
      n=backpackscaletouchy();
      if(n<0){return -1;}
      return backpackwidth-n;                
      break;
    case 2:
      n=backpackscaletouchx();
      if(n<0){return -1;}
      return n;                
      break;
    case 3:
      n=backpackscaletouchy();
      if(n<0){return -1;}
      return n;                
      break;
    case 4:
      n=backpackscaletouchx();
      if(n<0){return -1;}
      return backpackwidth-n;                
      break;
    default:
      return -1;
  }  
}


int backpacktouchy(){
  int n;
  switch(backpackrotation){
    case 1:
      n=backpackscaletouchx();
      if(n<0){return -1;}
      return n;                
      break;
    case 2:
      n=backpackscaletouchy();
      if(n<0){return -1;}
      return n;                
      break;
    case 3:
      n=backpackscaletouchx();
      if(n<0){return -1;}
      return backpackheight-n;                
      break;
    case 4:
      n=backpackscaletouchy();
      if(n<0){return -1;}
      return backpackheight-n;                
      break;
    default:
      return -1;
  }  
}


int backpackcharfont(int x, int y, char c, unsigned int f, unsigned int b, const char* font){
  int w,h,c0,c1,p,p0;
  char d;
  w=font[0];
  h=font[1];
  c0=font[2];
  c1=font[3]+c0;
  if(c<c0){return 0;}   //out of range
  if(c>c1){return 0;}   //out of range
  p0=4+(w*h*(c-c0))/8;
  backpackareaset(x,y,x+w-1,y+h-1);    //set area
  BACKPACK_CS=0;
  for(p=0;p<w*h;p++){
    if((p&7)==0){
      d=font[p0+(p/8)];
    }
    if(d&128){
      backpackdata(f>>16);backpackdata(f>>8);backpackdata(f);
    }else{
      backpackdata(b>>8);backpackdata(b>>8);backpackdata(b);
    }
    d=d<<1;
  }
  BACKPACK_CS=1;
  return w;             //allows arrays to be printed in a row
    

}

void backpackchararrayfont(int x0, int y0, const char *c, unsigned int f, unsigned int b, const char* font){
  while(*c){
    x0=x0+backpackcharfont(x0,y0,*c++,f,b,font);
    if(x0>backpackwidth){x0=0;}      //wrap around (will probably look ugly)
  }  


}

long map(long x, long in_min, long in_max, long out_min, long out_max)
{
  return (x - in_min) * (out_max - out_min) / (in_max - in_min) + out_min;
}

void backpackfcircle(int xo,int yo,int r,unsigned int c){ //https://en.wikipedia.org/wiki/Midpoint_circle_algorithm
  int e=0;
  int x=r;
  int y=0;
  while(x>=y){
    backpackvline(xo-y,yo+x,yo-x,c);
    backpackvline(xo+y,yo+x,yo-x,c);
    e=e+1+2*y;
    y=y+1;
    if(2*(e-x)+1>0){
      y=y-1;
      backpackvline(xo-x,yo-y,yo+y,c);
      backpackvline(xo+x,yo-y,yo+y,c);
      y=y+1;
      x=x-1;
      e=e+1-2*x;
    }
  }  
}

void backpackcircle(int xo,int yo,int r,unsigned int c){ //https://en.wikipedia.org/wiki/Midpoint_circle_algorithm
  int e=0;
  int x=r;
  int y=0;
  while(x>=y){
    backpackpoint(xo+x,yo+y,c);
    backpackpoint(xo-x,yo+y,c);
    backpackpoint(xo+x,yo-y,c);
    backpackpoint(xo-x,yo-y,c);
    backpackpoint(xo+y,yo+x,c);
    backpackpoint(xo-y,yo+x,c);
    backpackpoint(xo+y,yo-x,c);
    backpackpoint(xo-y,yo-x,c);
    e=e+1+2*y;
    y=y+1;
    if(2*(e-x)+1>0){
      x=x-1;
      e=e+1-2*x;
    }
  }
}

void backpackarc(int xo,int yo,int r,int mask, unsigned int c){ //https://en.wikipedia.org/wiki/Midpoint_circle_algorithm
  int e=0;
  int x=r;
  int y=0;
  while(x>=y){
    if(mask&4){backpackpoint(xo+x,yo+y,c);}
    if(mask&32){backpackpoint(xo-x,yo+y,c);}
    if(mask&2){backpackpoint(xo+x,yo-y,c);}
    if(mask&64){backpackpoint(xo-x,yo-y,c);}
    if(mask&8){backpackpoint(xo+y,yo+x,c);}
    if(mask&16){backpackpoint(xo-y,yo+x,c);}
    if(mask&1){backpackpoint(xo+y,yo-x,c);}
    if(mask&128){backpackpoint(xo-y,yo-x,c);}
    e=e+1+2*y;
    y=y+1;
    if(2*(e-x)+1>0){
      x=x-1;
      e=e+1-2*x;
    }
  }
}


void backpacktriangle(int x1,int y1,int x2,int y2,int x3,int y3,unsigned int c){ //custom Bresenham line algorithm
  //sort values, y1 at top
  int y;
  if(y1>y2){int i=y1;y1=y2;y2=i;i=x1;x1=x2;x2=i;}
  if(y2>y3){int i=y2;y2=y3;y3=i;i=x2;x2=x3;x3=i;}
  if(y1>y2){int i=y1;y1=y2;y2=i;i=x1;x1=x2;x2=i;}
  if(y1==y3){backpackhline(min(x1,min(x2,x3)),y1,max(x1,max(x2,x3)),c);return;}
  if(y1!=y2){
    int dy1=y2-y1;
    int dy2=y3-y1;
    int dx1=x2-x1;
    int dx2=x3-x1;
    int xa,xb;
    xa=x1*dy1-(dx1);
    xb=x1*dy2-(dx2);
    for(y=y1;y<y2;y++){
     xa=xa+dx1;
     xb=xb+dx2;
     backpackhline(xa/dy1,y,xb/dy2,c);
    }
   xb=xb+dx2;
   backpackhline(x2,y2,xb/dy2,c);        
  }
  if(y2!=y3){
    int dy1=y2-y3;
    int dy2=y1-y3;
    int dx1=x2-x3;
    int dx2=x1-x3;
    int xa,xb;
    xa=x3*dy1+(dx1);
    xb=x3*dy2+(dx2);
    for(y=y3;y>y2;y--){
     xa=xa-dx1;
     xb=xb-dx2;
     backpackhline(xa/dy1,y,xb/dy2,c);
    }        
   if(y1==y2){
    xb=xb+dx2;
    backpackhline(x2,y2,xb/dy2,c);        
   }
  }
}

void backpackrbox(int x1,int y1,int x2,int y2,int r, unsigned int c){  //outline only
    if(x2<x1){int i=x1;x1=x2;x2=i;}
    if(y2<y1){int i=y1;y1=y2;y2=i;}
    if(x2-x1<r*2){r=(x2-x1)/2;}
    if(y2-y1<r*2){r=(y2-y1)/2;}     //if arc would be larger than side
    backpackarc(x1+r,y1+r,r,192,c);     //top left
    backpackarc(x1+r,y2-r,r,48,c);      //bottom left
    backpackarc(x2-r,y1+r,r,3,c);       //top right
    backpackarc(x2-r,y2-r,r,12,c);      //bottom right
    backpackhline(x1+r,y1,x2-r,c);      //top line
    backpackhline(x1+r,y2,x2-r,c);      //bottom line
    backpackvline(x1,y1+r,y2-r,c);      //left line
    backpackvline(x2,y1+r,y2-r,c);      //right line    
}

void backpackrfbox(int x1,int y1,int x2,int y2,int r, unsigned int c){  //filled
    if(x2<x1){int i=x1;x1=x2;x2=i;}
    if(y2<y1){int i=y1;y1=y2;y2=i;}
    if(x2-x1<r*2){r=(x2-x1)/2;}
    if(y2-y1<r*2){r=(y2-y1)/2;}     //if arc would be larger than side
    backpackquad(x1+r,y1+r,r,8,c);     //top left
    backpackquad(x1+r,y2-r,r,4,c);      //bottom left
    backpackquad(x2-r,y1+r,r,1,c);       //top right
    backpackquad(x2-r,y2-r,r,2,c);      //bottom right
    backpackbox(x1+r,y1,x2-r,y1+r,c);   //top section
    backpackbox(x1,y1+r,x2,y2-r,c);     //middle section
    backpackbox(x1+r,y2-r,x2-r,y2,c);   //bottom section    
}

void backpackquad(int xo,int yo,int r,int mask, unsigned int c){ //https://en.wikipedia.org/wiki/Midpoint_circle_algorithm
  int e=0;
  int x=r;
  int y=0;
  while(x>=y){
    if(mask&1){backpackvline(xo+y,yo,yo-x,c);}
    if(mask&2){backpackvline(xo+y,yo+x,yo,c);}
    if(mask&4){backpackvline(xo-y,yo+x,yo,c);}
    if(mask&8){backpackvline(xo-y,yo,yo-x,c);}
    e=e+1+2*y;
    y=y+1;
    if(2*(e-x)+1>0){
      y=y-1;
      if(mask&1){backpackvline(xo+x,yo-y,yo,c);}
      if(mask&2){backpackvline(xo+x,yo,yo+y,c);}
      if(mask&4){backpackvline(xo-x,yo,yo+y,c);}
      if(mask&8){backpackvline(xo-x,yo-y,yo,c);}
      y=y+1;
      x=x-1;
      e=e+1-2*x;
    }
  }  
}

void backpacktextbutton(int x1,int y1,int x2,int y2,int r, const char *c,unsigned int f,unsigned int b,const char* font){  
//micromite style rounded text button
//position- from (x1,y1) to (x2,y2)
//corner radius r, string c, fg colour (text and border) f, bg colour b, font
  int w,h;
  w=font[0]*strlen(c);    
  h=font[1];
  backpackrfbox(x1,y1,x2,y2,r,b);       //background fill
  backpackrbox(x1,y1,x2,y2,r,f);        //border
  backpackchararrayfont((x1+x2-w)/2,(y1+y2-h)/2,c,f,b,font);
}

void backpacksetbuttonobject(struct backpackbutton *b,int x0,int y0,int x1,int y1, const char *text){
    b->x0=x0;
    b->y0=y0;
    b->x1=x1;
    b->y1=y1;
    b->text=text;  
}

bool backpackbuttonobjecttouched(struct backpackbutton *b){
    int x;
    x=backpacktouchx();
    if(x<b->x0){return 0;}
    if(x>b->x1){return 0;}
    x=backpacktouchy();
    if(x<b->y0){return 0;}
    if(x>b->y1){return 0;}
    return 1;
}

void backpackdrawbuttonobject(struct backpackbutton *b, bool pressed, int r, unsigned int f,unsigned int bg,const char* font){
    if(pressed){    //draw reversed
        backpacktextbutton(b->x0,b->y0,b->x1,b->y1,r,b->text,bg,f,font); 
    }else{          //draw normal
        backpacktextbutton(b->x0,b->y0,b->x1,b->y1,r,b->text,f,bg,font);         
    }
}

int backpackcheckbuttons(struct backpackbutton *b, int start, int end){
    int i;
    for(i=start;i<=end;i++){
        if(backpackbuttonobjecttouched(b+i)){
            return i;
        }
    }
    return -1;
}

void buttonStyle(struct backpackbutton *b, bool pressed){
    backpackdrawbuttonobject(b, pressed, BUTTON_RADIUS, BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);
}

//just does same as above but with background colour only
void buttonErase(struct backpackbutton *b, bool pressed){
    backpackdrawbuttonobject(b, pressed, BUTTON_RADIUS, BUTTON_BACK,BUTTON_BACK,BUTTON_FONT);    
}

void backpackmonobitmap(int x, int y, const uint8_t* bm){
    unsigned int w,h,p,t,d;
    w=bm[0];
    h=bm[1];
    t=2;
    backpackareaset(x,y,x+w-1,y+h-1);
    BACKPACK_CS=0;
    for(p=0;p<w*h;p++){
        d=bm[t]*256+bm[t+1];
        backpackdata((d&0xF800)>>8);backpackdata((d&0x7E0)>>3);backpackdata((d&0x1F)<<3);
        t=t+2;
    }
    BACKPACK_CS=1;    
}

void backpackmonobitmapscale(int x, int y, const uint8_t* bm,int s){
    unsigned int w,h,py,px,pz,t,d;
    w=bm[0];
    h=bm[1];
    backpackareaset(x,y,x+w*s-1,y+h*s-1);
    BACKPACK_CS=0;
    for(py=0;py<h*s;py++){
        for(px=0;px<w;px++){
            t=((py/s)*w+px)*2+2;
            d=bm[t]*256+bm[t+1];
            for(pz=0;pz<s;pz++){
                backpackdata((d&0xF800)>>8);backpackdata((d&0x7E0)>>3);backpackdata((d&0x1F)<<3);
            }
        }
    }
    BACKPACK_CS=1;    
}

void backpacknint(int x, int y, unsigned int n, int d, unsigned int f,unsigned int bg,const char* font){  //show n to d digits
    char zBlank=1;
    long t=1;
    int i;
    int s=0;
    for(i=0;i<d;i++){
        t=t*10;
    }
    while(n>=t){n=n-t;zBlank=0;}     //cap to d digits, no zBlank
    t=t/10;
    if(t==0){return;}
    while(t>0){
        while(n>=t){
            s=s+1;
            n=n-t;
        }
        if((s==0)&zBlank&&(t>1)){   //blank on leading zeroes, except last position
            x=x+backpackcharfont(x,y,' ',f,bg,font);
        }else{
            x=x+backpackcharfont(x,y,s+'0',f,bg,font);
            zBlank=0;
        }
        s=0;
        t=t/10;
    }
}

void backpackshowms(int x, int y, unsigned int n, unsigned int f,unsigned int bg,const char* font){  //show time in m:s to d digits)
    unsigned int m=n/60;
    unsigned int s=n-(m*60);
    if(m>999){
        backpackchararrayfont(x,y,"OVF",f,bg,font);        
    }else{
        backpacknint(x,y,m,3,f,bg,font);
    }
    backpackcharfont(x+48,y,':',f,bg,font);
    backpackcharfont(x+64,y,s/10+'0',f,bg,font);
    backpackcharfont(x+80,y,s%10+'0',f,bg,font);        
}